<?php
/**
 * صفحه تیکت‌های کاربر
 */
function renderUserTickets() {
?>
<main class="pb-20 lg:pb-8 pt-16 min-h-screen">
    <div class="p-4 lg:p-8 max-w-4xl mx-auto">
        <!-- هدر -->
        <div class="flex items-center justify-between mb-6">
            <div>
                <h1 class="text-2xl font-bold">تیکت‌های پشتیبانی</h1>
                <p class="text-gray-400 mt-1">ارتباط با تیم پشتیبانی</p>
            </div>
            <button onclick="openNewTicketModal()" class="bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl px-6 py-2.5 font-medium hover:opacity-90 transition-opacity flex items-center gap-2">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                </svg>
                تیکت جدید
            </button>
        </div>

        <!-- لیست تیکت‌ها -->
        <div class="space-y-3" id="ticketsList">
            <!-- تیکت‌ها با AJAX لود می‌شوند -->
        </div>
        
        <div id="noTickets" class="hidden glass rounded-3xl p-12 text-center">
            <svg class="w-20 h-20 mx-auto mb-4 text-gray-400 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 10h.01M12 10h.01M16 10h.01M9 16H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-5l-5 5v-5z"></path>
            </svg>
            <h3 class="text-xl font-bold mb-2">تیکتی وجود ندارد</h3>
            <p class="text-gray-400 mb-4">برای ارتباط با پشتیبانی، تیکت جدید ایجاد کنید</p>
            <button onclick="openNewTicketModal()" class="bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl px-6 py-2.5 font-medium hover:opacity-90">
                ایجاد تیکت جدید
            </button>
        </div>
    </div>
</main>

<!-- Modal تیکت جدید -->
<div id="newTicketModal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 bg-black/50 backdrop-blur-sm">
    <div class="glass rounded-3xl p-6 w-full max-w-lg animate-scale-in">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-lg font-bold">تیکت جدید</h3>
            <button onclick="closeNewTicketModal()" class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-500/20">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        
        <form id="newTicketForm" class="space-y-4">
            <div>
                <label class="block text-sm text-gray-400 mb-2">بخش</label>
                <select name="department" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                    <option value="technical">فنی</option>
                    <option value="sales">فروش</option>
                    <option value="financial">مالی</option>
                    <option value="general">عمومی</option>
                </select>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">اولویت</label>
                <select name="priority" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                    <option value="low">پایین</option>
                    <option value="medium" selected>متوسط</option>
                    <option value="high">بالا</option>
                </select>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">موضوع</label>
                <input type="text" name="subject" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" placeholder="موضوع تیکت" required>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">پیام</label>
                <textarea name="message" rows="5" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none resize-none" placeholder="متن پیام خود را بنویسید..." required></textarea>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">پیوست (اختیاری)</label>
                <input type="file" name="attachment" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:bg-primary-500 file:text-white" accept="image/*,.pdf,.zip">
            </div>
            
            <button type="submit" class="w-full bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl py-3 font-medium hover:opacity-90 transition-opacity">
                ارسال تیکت
            </button>
        </form>
    </div>
</div>

<!-- Modal مشاهده تیکت -->
<div id="viewTicketModal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 bg-black/50 backdrop-blur-sm">
    <div class="glass rounded-3xl w-full max-w-2xl animate-scale-in max-h-[90vh] flex flex-col">
        <!-- هدر -->
        <div class="p-6 border-b border-white/10 flex items-center justify-between shrink-0">
            <div>
                <h3 class="text-lg font-bold" id="viewTicketSubject">موضوع تیکت</h3>
                <div class="flex items-center gap-2 mt-1">
                    <span id="viewTicketStatus" class="px-2 py-0.5 rounded text-xs bg-blue-500/20 text-blue-400">باز</span>
                    <span class="text-gray-400 text-sm" id="viewTicketDate"></span>
                </div>
            </div>
            <button onclick="closeViewTicketModal()" class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-500/20">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        
        <!-- پیام‌ها -->
        <div class="flex-1 overflow-y-auto p-6 space-y-4" id="ticketMessages">
            <!-- پیام‌ها با AJAX لود می‌شوند -->
        </div>
        
        <!-- فرم پاسخ -->
        <div id="ticketReplySection" class="p-6 border-t border-white/10 shrink-0">
            <form id="ticketReplyForm" class="space-y-3">
                <input type="hidden" name="ticket_id" id="replyTicketId">
                <textarea name="message" rows="3" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none resize-none" placeholder="پاسخ خود را بنویسید..." required></textarea>
                <div class="flex items-center justify-between">
                    <label class="flex items-center gap-2 cursor-pointer text-gray-400 hover:text-white">
                        <input type="file" name="attachment" class="hidden" accept="image/*,.pdf">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13"></path>
                        </svg>
                        پیوست
                    </label>
                    <button type="submit" class="bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl px-6 py-2.5 font-medium hover:opacity-90 transition-opacity">
                        ارسال پاسخ
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php renderMobileNav(); ?>

<script>
let currentTicketId = null;

// لود تیکت‌ها
async function loadTickets() {
    try {
        const response = await fetch('api.php?action=user_get_tickets');
        const data = await response.json();
        
        if (data.success) {
            renderTickets(data.tickets);
        }
    } catch (error) {
        console.error('Error loading tickets:', error);
    }
}

function renderTickets(tickets) {
    const container = document.getElementById('ticketsList');
    const noTickets = document.getElementById('noTickets');
    
    if (tickets.length === 0) {
        container.innerHTML = '';
        noTickets.classList.remove('hidden');
        return;
    }
    
    noTickets.classList.add('hidden');
    container.innerHTML = tickets.map(ticket => `
        <div class="glass rounded-2xl p-4 hover:bg-white/5 cursor-pointer transition-all" onclick="viewTicket(${ticket.id})">
            <div class="flex items-center justify-between">
                <div class="flex items-center gap-3">
                    <div class="w-12 h-12 rounded-xl ${getPriorityBg(ticket.priority)} flex items-center justify-center">
                        <svg class="w-6 h-6 ${getPriorityColor(ticket.priority)}" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 8h10M7 12h4m1 8l-4-4H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-3l-4 4z"></path>
                        </svg>
                    </div>
                    <div>
                        <div class="font-medium">${ticket.subject}</div>
                        <div class="text-sm text-gray-400">#${ticket.id} • ${getDepartmentText(ticket.department)}</div>
                    </div>
                </div>
                <div class="text-left">
                    <span class="px-3 py-1 rounded-full text-xs ${getTicketStatusClass(ticket.status)}">
                        ${getTicketStatusText(ticket.status)}
                    </span>
                    <div class="text-xs text-gray-500 mt-1">${ticket.updated_at}</div>
                </div>
            </div>
        </div>
    `).join('');
}

// Modal تیکت جدید
function openNewTicketModal() {
    document.getElementById('newTicketForm').reset();
    document.getElementById('newTicketModal').classList.remove('hidden');
    document.getElementById('newTicketModal').classList.add('flex');
}

function closeNewTicketModal() {
    document.getElementById('newTicketModal').classList.add('hidden');
    document.getElementById('newTicketModal').classList.remove('flex');
}

// ارسال تیکت جدید
document.getElementById('newTicketForm')?.addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', 'create_ticket');
    
    try {
        const response = await fetch('api.php', { method: 'POST', body: formData });
        const data = await response.json();
        
        if (data.success) {
            closeNewTicketModal();
            showToast('تیکت با موفقیت ارسال شد', 'success');
            loadTickets();
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در ارسال تیکت', 'error');
    }
});

// مشاهده تیکت
async function viewTicket(ticketId) {
    currentTicketId = ticketId;
    
    try {
        const response = await fetch(`api.php?action=user_get_ticket&id=${ticketId}`);
        const data = await response.json();
        
        if (data.success) {
            const ticket = data.ticket;
            document.getElementById('viewTicketSubject').textContent = ticket.subject;
            document.getElementById('viewTicketStatus').textContent = getTicketStatusText(ticket.status);
            document.getElementById('viewTicketStatus').className = `px-2 py-0.5 rounded text-xs ${getTicketStatusClass(ticket.status)}`;
            document.getElementById('viewTicketDate').textContent = ticket.created_at;
            document.getElementById('replyTicketId').value = ticket.id;
            
            // نمایش/مخفی فرم پاسخ
            if (ticket.status === 'closed') {
                document.getElementById('ticketReplySection').classList.add('hidden');
            } else {
                document.getElementById('ticketReplySection').classList.remove('hidden');
            }
            
            renderTicketMessages(ticket.messages);
            
            document.getElementById('viewTicketModal').classList.remove('hidden');
            document.getElementById('viewTicketModal').classList.add('flex');
        }
    } catch (error) {
        showToast('خطا در دریافت تیکت', 'error');
    }
}

function renderTicketMessages(messages) {
    const container = document.getElementById('ticketMessages');
    container.innerHTML = messages.map(msg => `
        <div class="flex ${msg.is_admin ? 'justify-start' : 'justify-end'}">
            <div class="max-w-[85%] ${msg.is_admin ? 'bg-primary-500/20' : 'bg-white/10'} rounded-2xl p-4">
                <div class="flex items-center gap-2 mb-2">
                    <div class="w-8 h-8 rounded-full ${msg.is_admin ? 'bg-primary-500' : 'bg-gray-500'} flex items-center justify-center text-sm font-bold">
                        ${msg.is_admin ? '🛡️' : '👤'}
                    </div>
                    <div>
                        <div class="font-medium text-sm">${msg.is_admin ? 'پشتیبانی' : 'شما'}</div>
                        <div class="text-xs text-gray-400">${msg.created_at}</div>
                    </div>
                </div>
                <div class="text-sm leading-relaxed whitespace-pre-wrap">${msg.message}</div>
                ${msg.attachment ? `
                    <div class="mt-3">
                        <a href="${msg.attachment}" target="_blank" class="flex items-center gap-2 text-primary-400 hover:underline text-sm">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13"></path>
                            </svg>
                            مشاهده پیوست
                        </a>
                    </div>
                ` : ''}
            </div>
        </div>
    `).join('');
    
    container.scrollTop = container.scrollHeight;
}

function closeViewTicketModal() {
    document.getElementById('viewTicketModal').classList.add('hidden');
    document.getElementById('viewTicketModal').classList.remove('flex');
    currentTicketId = null;
}

// ارسال پاسخ
document.getElementById('ticketReplyForm')?.addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', 'reply_ticket');
    
    try {
        const response = await fetch('api.php', { method: 'POST', body: formData });
        const data = await response.json();
        
        if (data.success) {
            this.reset();
            document.getElementById('replyTicketId').value = currentTicketId;
            viewTicket(currentTicketId);
            showToast('پاسخ ارسال شد', 'success');
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در ارسال پاسخ', 'error');
    }
});

// توابع کمکی
function getPriorityBg(priority) {
    const bgs = { 'high': 'bg-red-500/20', 'medium': 'bg-yellow-500/20', 'low': 'bg-green-500/20' };
    return bgs[priority] || bgs['medium'];
}

function getPriorityColor(priority) {
    const colors = { 'high': 'text-red-400', 'medium': 'text-yellow-400', 'low': 'text-green-400' };
    return colors[priority] || colors['medium'];
}

function getDepartmentText(dept) {
    const texts = { 'technical': 'فنی', 'sales': 'فروش', 'financial': 'مالی', 'general': 'عمومی' };
    return texts[dept] || dept;
}

function getTicketStatusClass(status) {
    const classes = {
        'open': 'bg-blue-500/20 text-blue-400',
        'waiting': 'bg-yellow-500/20 text-yellow-400',
        'answered': 'bg-green-500/20 text-green-400',
        'closed': 'bg-gray-500/20 text-gray-400'
    };
    return classes[status] || classes['open'];
}

function getTicketStatusText(status) {
    const texts = { 'open': 'باز', 'waiting': 'در انتظار', 'answered': 'پاسخ داده شده', 'closed': 'بسته' };
    return texts[status] || status;
}

// لود اولیه
document.addEventListener('DOMContentLoaded', loadTickets);
</script>
<?php
}
?>