<?php
/**
 * صفحه مدیریت کاربران
 */
function renderAdminUsers() {
?>
<main class="pr-0 lg:pr-64 pt-16 min-h-screen">
    <div class="p-4 lg:p-8">
        <!-- هدر صفحه -->
        <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4 mb-8">
            <div>
                <h1 class="text-2xl font-bold">مدیریت کاربران</h1>
                <p class="text-gray-400 mt-1">مشاهده و مدیریت کاربران سایت</p>
            </div>
            <div class="flex items-center gap-3">
                <button onclick="openAddUserModal()" class="bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl px-6 py-2.5 font-medium hover:opacity-90 transition-opacity flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                    </svg>
                    افزودن کاربر
                </button>
                <button onclick="exportUsers()" class="glass rounded-xl px-4 py-2.5 hover:bg-white/10 transition-all flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                    </svg>
                    خروجی اکسل
                </button>
            </div>
        </div>

        <!-- آمار کاربران -->
        <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
            <div class="glass rounded-2xl p-4">
                <div class="flex items-center gap-3">
                    <div class="w-12 h-12 rounded-xl bg-blue-500/20 flex items-center justify-center">
                        <svg class="w-6 h-6 text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"></path>
                        </svg>
                    </div>
                    <div>
                        <div class="text-2xl font-bold" id="totalUsersCount">0</div>
                        <div class="text-sm text-gray-400">کل کاربران</div>
                    </div>
                </div>
            </div>
            <div class="glass rounded-2xl p-4">
                <div class="flex items-center gap-3">
                    <div class="w-12 h-12 rounded-xl bg-green-500/20 flex items-center justify-center">
                        <svg class="w-6 h-6 text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                    </div>
                    <div>
                        <div class="text-2xl font-bold" id="activeUsersCount">0</div>
                        <div class="text-sm text-gray-400">کاربران فعال</div>
                    </div>
                </div>
            </div>
            <div class="glass rounded-2xl p-4">
                <div class="flex items-center gap-3">
                    <div class="w-12 h-12 rounded-xl bg-red-500/20 flex items-center justify-center">
                        <svg class="w-6 h-6 text-red-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M18.364 18.364A9 9 0 005.636 5.636m12.728 12.728A9 9 0 015.636 5.636m12.728 12.728L5.636 5.636"></path>
                        </svg>
                    </div>
                    <div>
                        <div class="text-2xl font-bold" id="blockedUsersCount">0</div>
                        <div class="text-sm text-gray-400">مسدود شده</div>
                    </div>
                </div>
            </div>
            <div class="glass rounded-2xl p-4">
                <div class="flex items-center gap-3">
                    <div class="w-12 h-12 rounded-xl bg-yellow-500/20 flex items-center justify-center">
                        <svg class="w-6 h-6 text-yellow-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                    </div>
                    <div>
                        <div class="text-2xl font-bold" id="todayUsersCount">0</div>
                        <div class="text-sm text-gray-400">ثبت‌نام امروز</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- فیلترها و جستجو -->
        <div class="glass rounded-3xl p-6 mb-6">
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div class="relative">
                    <input type="text" id="searchUsers" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 pr-12 focus:border-primary-500 outline-none" placeholder="جستجو (نام، ایمیل، موبایل)...">
                    <svg class="w-5 h-5 absolute right-4 top-1/2 -translate-y-1/2 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                    </svg>
                </div>
                <select id="userStatusFilter" class="bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <option value="">همه وضعیت‌ها</option>
                    <option value="active">فعال</option>
                    <option value="blocked">مسدود</option>
                    <option value="pending">در انتظار تایید</option>
                </select>
                <select id="userSortFilter" class="bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <option value="newest">جدیدترین</option>
                    <option value="oldest">قدیمی‌ترین</option>
                    <option value="most_orders">بیشترین سفارش</option>
                    <option value="highest_balance">بیشترین موجودی</option>
                </select>
                <select id="userRoleFilter" class="bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <option value="">همه نقش‌ها</option>
                    <option value="user">کاربر عادی</option>
                    <option value="vip">VIP</option>
                    <option value="reseller">نمایندگی</option>
                </select>
            </div>
        </div>

        <!-- جدول کاربران -->
        <div class="glass rounded-3xl overflow-hidden">
            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead>
                        <tr class="text-gray-400 text-sm border-b border-white/10">
                            <th class="text-right p-4">
                                <input type="checkbox" id="selectAllUsers" class="rounded bg-white/10 border-white/20">
                            </th>
                            <th class="text-right p-4">کاربر</th>
                            <th class="text-right p-4">شماره موبایل</th>
                            <th class="text-right p-4">موجودی</th>
                            <th class="text-right p-4">کانفیگ فعال</th>
                            <th class="text-right p-4">وضعیت</th>
                            <th class="text-right p-4">تاریخ عضویت</th>
                            <th class="text-center p-4">عملیات</th>
                        </tr>
                    </thead>
                    <tbody id="usersTable">
                        <!-- داده‌ها با AJAX لود می‌شوند -->
                    </tbody>
                </table>
            </div>
            
            <!-- صفحه‌بندی -->
            <div class="p-4 border-t border-white/10">
                <div class="flex flex-col md:flex-row items-center justify-between gap-4">
                    <div class="flex items-center gap-4">
                        <span class="text-gray-400 text-sm" id="usersCount">نمایش 0 کاربر</span>
                        <div class="flex items-center gap-2">
                            <button onclick="bulkAction('block')" class="px-3 py-1.5 rounded-lg bg-red-500/20 text-red-400 text-sm hover:bg-red-500/30">مسدود کردن</button>
                            <button onclick="bulkAction('activate')" class="px-3 py-1.5 rounded-lg bg-green-500/20 text-green-400 text-sm hover:bg-green-500/30">فعال‌سازی</button>
                            <button onclick="bulkAction('delete')" class="px-3 py-1.5 rounded-lg bg-gray-500/20 text-gray-400 text-sm hover:bg-gray-500/30">حذف</button>
                        </div>
                    </div>
                    <div class="flex items-center gap-2" id="usersPagination">
                        <!-- صفحه‌بندی -->
                    </div>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- Modal افزودن/ویرایش کاربر -->
<div id="userModal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 bg-black/50 backdrop-blur-sm">
    <div class="glass rounded-3xl p-6 w-full max-w-lg animate-scale-in max-h-[90vh] overflow-y-auto">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-lg font-bold" id="userModalTitle">افزودن کاربر جدید</h3>
            <button onclick="closeUserModal()" class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-500/20">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        <form id="userForm" class="space-y-4">
            <input type="hidden" name="user_id" id="editUserId">
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm text-gray-400 mb-2">نام کاربری <span class="text-red-400">*</span></label>
                    <input type="text" name="username" id="userUsername" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">ایمیل</label>
                    <input type="email" name="email" id="userEmail" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                </div>
            </div>
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm text-gray-400 mb-2">شماره موبایل <span class="text-red-400">*</span></label>
                    <input type="tel" name="phone" id="userPhone" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">رمز عبور <span class="text-red-400 new-user-only">*</span></label>
                    <input type="password" name="password" id="userPassword" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <p class="text-xs text-gray-500 mt-1 edit-mode-only hidden">خالی بگذارید تا تغییر نکند</p>
                </div>
            </div>
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm text-gray-400 mb-2">نقش کاربر</label>
                    <select name="role" id="userRole" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                        <option value="user">کاربر عادی</option>
                        <option value="vip">VIP</option>
                        <option value="reseller">نمایندگی</option>
                        <option value="admin">مدیر</option>
                    </select>
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">وضعیت</label>
                    <select name="status" id="userStatus" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                        <option value="active">فعال</option>
                        <option value="blocked">مسدود</option>
                        <option value="pending">در انتظار تایید</option>
                    </select>
                </div>
            </div>
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm text-gray-400 mb-2">موجودی کیف پول (تومان)</label>
                    <input type="number" name="balance" id="userBalance" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" value="0">
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">حداکثر دستگاه</label>
                    <input type="number" name="max_devices" id="userMaxDevices" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" value="2">
                </div>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">یادداشت ادمین</label>
                <textarea name="admin_note" id="userAdminNote" rows="2" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none resize-none"></textarea>
            </div>
            
            <div class="flex gap-3 pt-4">
                <button type="submit" class="flex-1 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl py-3 font-medium hover:opacity-90 transition-opacity">
                    ذخیره
                </button>
                <button type="button" onclick="closeUserModal()" class="flex-1 glass rounded-xl py-3 font-medium hover:bg-white/10">
                    انصراف
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Modal جزئیات کاربر -->
<div id="userDetailModal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 bg-black/50 backdrop-blur-sm">
    <div class="glass rounded-3xl p-6 w-full max-w-4xl animate-scale-in max-h-[90vh] overflow-y-auto">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-lg font-bold">جزئیات کاربر</h3>
            <button onclick="closeUserDetailModal()" class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-500/20">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        <div id="userDetailContent">
            <!-- محتوا با AJAX لود می‌شود -->
        </div>
    </div>
</div>

<!-- Modal تغییر موجودی -->
<div id="balanceModal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 bg-black/50 backdrop-blur-sm">
    <div class="glass rounded-3xl p-6 w-full max-w-md animate-scale-in">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-lg font-bold">تغییر موجودی کیف پول</h3>
            <button onclick="closeBalanceModal()" class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-500/20">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        <form id="balanceForm" class="space-y-4">
            <input type="hidden" name="user_id" id="balanceUserId">
            
            <div class="text-center mb-4">
                <p class="text-gray-400">کاربر: <span id="balanceUsername" class="text-white font-medium"></span></p>
                <p class="text-gray-400">موجودی فعلی: <span id="currentBalance" class="text-primary-400 font-bold"></span> تومان</p>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">نوع عملیات</label>
                <div class="grid grid-cols-2 gap-3">
                    <label class="glass rounded-xl p-3 cursor-pointer border-2 border-transparent has-[:checked]:border-green-500">
                        <input type="radio" name="balance_type" value="add" class="hidden" checked>
                        <div class="flex items-center justify-center gap-2 text-green-400">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                            </svg>
                            افزایش
                        </div>
                    </label>
                    <label class="glass rounded-xl p-3 cursor-pointer border-2 border-transparent has-[:checked]:border-red-500">
                        <input type="radio" name="balance_type" value="subtract" class="hidden">
                        <div class="flex items-center justify-center gap-2 text-red-400">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4"></path>
                            </svg>
                            کاهش
                        </div>
                    </label>
                </div>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">مبلغ (تومان)</label>
                <input type="number" name="amount" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required min="1">
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">توضیحات</label>
                <input type="text" name="description" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" placeholder="مثال: پاداش معرفی">
            </div>
            
            <button type="submit" class="w-full bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl py-3 font-medium hover:opacity-90 transition-opacity">
                اعمال تغییرات
            </button>
        </form>
    </div>
</div>

<!-- Modal اختصاص کانفیگ -->
<div id="assignConfigModal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 bg-black/50 backdrop-blur-sm">
    <div class="glass rounded-3xl p-6 w-full max-w-md animate-scale-in">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-lg font-bold">اختصاص کانفیگ به کاربر</h3>
            <button onclick="closeAssignConfigModal()" class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-500/20">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        <form id="assignConfigForm" class="space-y-4">
            <input type="hidden" name="user_id" id="assignConfigUserId">
            
            <div class="text-center mb-4">
                <p class="text-gray-400">کاربر: <span id="assignConfigUsername" class="text-white font-medium"></span></p>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">انتخاب کانفیگ</label>
                <select name="config_id" id="configSelect" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                    <option value="">انتخاب کنید...</option>
                </select>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">مدت زمان (روز)</label>
                <input type="number" name="duration" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" value="30" min="1" required>
            </div>
            
            <div>
                <label class="flex items-center gap-2 cursor-pointer">
                    <input type="checkbox" name="notify_user" class="rounded bg-white/10 border-white/20" checked>
                    <span class="text-sm">اطلاع‌رسانی به کاربر</span>
                </label>
            </div>
            
            <button type="submit" class="w-full bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl py-3 font-medium hover:opacity-90 transition-opacity">
                اختصاص کانفیگ
            </button>
        </form>
    </div>
</div>

<script>
let currentUserPage = 1;
const usersPerPage = 20;
let selectedUsers = [];

// لود کاربران
async function loadUsers(page = 1) {
    currentUserPage = page;
    const status = document.getElementById('userStatusFilter').value;
    const sort = document.getElementById('userSortFilter').value;
    const role = document.getElementById('userRoleFilter').value;
    const search = document.getElementById('searchUsers').value;
    
    try {
        const response = await fetch(`api.php?action=admin_get_users&page=${page}&limit=${usersPerPage}&status=${status}&sort=${sort}&role=${role}&search=${encodeURIComponent(search)}`);
        const data = await response.json();
        
        if (data.success) {
            renderUsersTable(data.users);
            renderUsersPagination(data.total, page);
            document.getElementById('usersCount').textContent = `نمایش ${data.users.length} از ${data.total} کاربر`;
            
            // آمار
            document.getElementById('totalUsersCount').textContent = data.stats.total;
            document.getElementById('activeUsersCount').textContent = data.stats.active;
            document.getElementById('blockedUsersCount').textContent = data.stats.blocked;
            document.getElementById('todayUsersCount').textContent = data.stats.today;
        }
    } catch (error) {
        console.error('Error loading users:', error);
    }
}

function renderUsersTable(users) {
    const tbody = document.getElementById('usersTable');
    
    if (users.length === 0) {
        tbody.innerHTML = `
            <tr>
                <td colspan="8" class="text-center py-12 text-gray-400">
                    <svg class="w-16 h-16 mx-auto mb-4 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0z"></path>
                    </svg>
                    کاربری یافت نشد
                </td>
            </tr>
        `;
        return;
    }
    
    tbody.innerHTML = users.map(user => `
        <tr class="border-b border-white/5 hover:bg-white/5">
            <td class="p-4">
                <input type="checkbox" class="user-checkbox rounded bg-white/10 border-white/20" data-id="${user.id}" onchange="updateSelectedUsers()">
            </td>
            <td class="p-4">
                <div class="flex items-center gap-3">
                    <div class="w-10 h-10 rounded-full bg-gradient-to-r ${getRoleGradient(user.role)} flex items-center justify-center text-sm font-bold">
                        ${user.username.charAt(0).toUpperCase()}
                    </div>
                    <div>
                        <div class="font-medium flex items-center gap-2">
                            ${user.username}
                            ${user.role === 'vip' ? '<span class="px-1.5 py-0.5 rounded text-xs bg-yellow-500/20 text-yellow-400">VIP</span>' : ''}
                            ${user.role === 'reseller' ? '<span class="px-1.5 py-0.5 rounded text-xs bg-purple-500/20 text-purple-400">نماینده</span>' : ''}
                        </div>
                        <div class="text-xs text-gray-400">${user.email || ''}</div>
                    </div>
                </div>
            </td>
            <td class="p-4 font-mono text-sm">${user.phone || '-'}</td>
            <td class="p-4">
                <span class="text-primary-400 font-medium">${numberFormat(user.balance)}</span>
                <span class="text-xs text-gray-400">تومان</span>
            </td>
            <td class="p-4">
                <span class="px-2 py-1 rounded-lg bg-blue-500/20 text-blue-400 text-sm">${user.active_configs}</span>
            </td>
            <td class="p-4">
                <span class="px-3 py-1 rounded-full text-xs ${getUserStatusClass(user.status)}">
                    ${getUserStatusText(user.status)}
                </span>
            </td>
            <td class="p-4 text-gray-400 text-sm">${user.created_at}</td>
            <td class="p-4">
                <div class="flex items-center justify-center gap-1">
                    <button onclick="viewUserDetail(${user.id})" class="w-8 h-8 rounded-lg bg-blue-500/20 text-blue-400 flex items-center justify-center hover:bg-blue-500/30" title="جزئیات">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path>
                        </svg>
                    </button>
                    <button onclick="editUser(${user.id})" class="w-8 h-8 rounded-lg bg-yellow-500/20 text-yellow-400 flex items-center justify-center hover:bg-yellow-500/30" title="ویرایش">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                        </svg>
                    </button>
                    <button onclick="openBalanceModal(${user.id}, '${user.username}', ${user.balance})" class="w-8 h-8 rounded-lg bg-green-500/20 text-green-400 flex items-center justify-center hover:bg-green-500/30" title="موجودی">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                    </button>
                    <button onclick="openAssignConfigModal(${user.id}, '${user.username}')" class="w-8 h-8 rounded-lg bg-purple-500/20 text-purple-400 flex items-center justify-center hover:bg-purple-500/30" title="اختصاص کانفیگ">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13.828 10.172a4 4 0 00-5.656 0l-4 4a4 4 0 105.656 5.656l1.102-1.101m-.758-4.899a4 4 0 005.656 0l4-4a4 4 0 00-5.656-5.656l-1.1 1.1"></path>
                        </svg>
                    </button>
                    <button onclick="toggleUserStatus(${user.id}, '${user.status}')" class="w-8 h-8 rounded-lg ${user.status === 'blocked' ? 'bg-green-500/20 text-green-400' : 'bg-red-500/20 text-red-400'} flex items-center justify-center hover:opacity-80" title="${user.status === 'blocked' ? 'فعال‌سازی' : 'مسدود'}">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            ${user.status === 'blocked' ? 
                                '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>' :
                                '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M18.364 18.364A9 9 0 005.636 5.636m12.728 12.728A9 9 0 015.636 5.636m12.728 12.728L5.636 5.636"></path>'
                            }
                        </svg>
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Modal افزودن کاربر
function openAddUserModal() {
    document.getElementById('userModalTitle').textContent = 'افزودن کاربر جدید';
    document.getElementById('userForm').reset();
    document.getElementById('editUserId').value = '';
    document.querySelectorAll('.edit-mode-only').forEach(el => el.classList.add('hidden'));
    document.querySelectorAll('.new-user-only').forEach(el => el.style.display = 'inline');
    document.getElementById('userPassword').required = true;
    document.getElementById('userModal').classList.remove('hidden');
    document.getElementById('userModal').classList.add('flex');
}

function closeUserModal() {
    document.getElementById('userModal').classList.add('hidden');
    document.getElementById('userModal').classList.remove('flex');
}

// ویرایش کاربر
async function editUser(userId) {
    try {
        const response = await fetch(`api.php?action=admin_get_user&id=${userId}`);
        const data = await response.json();
        
        if (data.success) {
            const user = data.user;
            document.getElementById('userModalTitle').textContent = 'ویرایش کاربر';
            document.getElementById('editUserId').value = user.id;
            document.getElementById('userUsername').value = user.username;
            document.getElementById('userEmail').value = user.email || '';
            document.getElementById('userPhone').value = user.phone || '';
            document.getElementById('userRole').value = user.role;
            document.getElementById('userStatus').value = user.status;
            document.getElementById('userBalance').value = user.balance;
            document.getElementById('userMaxDevices').value = user.max_devices;
            document.getElementById('userAdminNote').value = user.admin_note || '';
            document.getElementById('userPassword').value = '';
            document.getElementById('userPassword').required = false;
            
            document.querySelectorAll('.edit-mode-only').forEach(el => el.classList.remove('hidden'));
            document.querySelectorAll('.new-user-only').forEach(el => el.style.display = 'none');
            
            document.getElementById('userModal').classList.remove('hidden');
            document.getElementById('userModal').classList.add('flex');
        }
    } catch (error) {
        showToast('خطا در دریافت اطلاعات کاربر', 'error');
    }
}

// ذخیره کاربر
document.getElementById('userForm')?.addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', formData.get('user_id') ? 'admin_update_user' : 'admin_add_user');
    
    try {
        const response = await fetch('api.php', { method: 'POST', body: formData });
        const data = await response.json();
        
        if (data.success) {
            showToast(data.message, 'success');
            closeUserModal();
            loadUsers(currentUserPage);
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در ذخیره اطلاعات', 'error');
    }
});

// مشاهده جزئیات کاربر
async function viewUserDetail(userId) {
    try {
        const response = await fetch(`api.php?action=admin_get_user_detail&id=${userId}`);
        const data = await response.json();
        
        if (data.success) {
            const user = data.user;
            document.getElementById('userDetailContent').innerHTML = `
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <!-- اطلاعات کاربر -->
                    <div class="glass rounded-2xl p-4">
                        <div class="flex items-center gap-4 mb-4">
                            <div class="w-16 h-16 rounded-full bg-gradient-to-r ${getRoleGradient(user.role)} flex items-center justify-center text-2xl font-bold">
                                ${user.username.charAt(0).toUpperCase()}
                            </div>
                            <div>
                                <h4 class="font-bold text-lg">${user.username}</h4>
                                <span class="px-2 py-1 rounded text-xs ${getUserStatusClass(user.status)}">${getUserStatusText(user.status)}</span>
                            </div>
                        </div>
                        <div class="space-y-2 text-sm">
                            <div class="flex justify-between">
                                <span class="text-gray-400">ایمیل:</span>
                                <span>${user.email || '-'}</span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-400">موبایل:</span>
                                <span class="font-mono">${user.phone || '-'}</span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-400">موجودی:</span>
                                <span class="text-primary-400 font-medium">${numberFormat(user.balance)} تومان</span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-400">تاریخ عضویت:</span>
                                <span>${user.created_at}</span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-400">آخرین ورود:</span>
                                <span>${user.last_login || '-'}</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- آمار -->
                    <div class="glass rounded-2xl p-4">
                        <h4 class="font-medium mb-4">آمار کاربر</h4>
                        <div class="space-y-3">
                            <div class="flex items-center justify-between p-3 bg-blue-500/10 rounded-xl">
                                <span>کل سفارشات</span>
                                <span class="font-bold text-blue-400">${user.total_orders}</span>
                            </div>
                            <div class="flex items-center justify-between p-3 bg-green-500/10 rounded-xl">
                                <span>کانفیگ فعال</span>
                                <span class="font-bold text-green-400">${user.active_configs}</span>
                            </div>
                            <div class="flex items-center justify-between p-3 bg-yellow-500/10 rounded-xl">
                                <span>مجموع پرداختی</span>
                                <span class="font-bold text-yellow-400">${numberFormat(user.total_paid)} تومان</span>
                            </div>
                            <div class="flex items-center justify-between p-3 bg-purple-500/10 rounded-xl">
                                <span>زیرمجموعه‌ها</span>
                                <span class="font-bold text-purple-400">${user.referral_count}</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- دستگاه‌های متصل -->
                    <div class="glass rounded-2xl p-4">
                        <h4 class="font-medium mb-4">دستگاه‌های متصل (${user.devices.length}/${user.max_devices})</h4>
                        <div class="space-y-2 max-h-48 overflow-y-auto">
                            ${user.devices.map(device => `
                                <div class="flex items-center justify-between p-2 bg-white/5 rounded-lg">
                                    <div class="flex items-center gap-2">
                                        <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 18h.01M8 21h8a2 2 0 002-2V5a2 2 0 00-2-2H8a2 2 0 00-2 2v14a2 2 0 002 2z"></path>
                                        </svg>
                                        <span class="text-sm">${device.name}</span>
                                    </div>
                                    <button onclick="removeDevice(${device.id})" class="text-red-400 hover:text-red-300">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                        </svg>
                                    </button>
                                </div>
                            `).join('') || '<p class="text-gray-400 text-sm">دستگاهی متصل نیست</p>'}
                        </div>
                    </div>
                </div>
                
                <!-- کانفیگ‌های فعال -->
                <div class="glass rounded-2xl p-4 mt-6">
                    <h4 class="font-medium mb-4">کانفیگ‌های فعال</h4>
                    <div class="overflow-x-auto">
                        <table class="w-full text-sm">
                            <thead>
                                <tr class="text-gray-400 border-b border-white/10">
                                    <th class="text-right pb-2">نام</th>
                                    <th class="text-right pb-2">نوع</th>
                                    <th class="text-right pb-2">لوکیشن</th>
                                    <th class="text-right pb-2">روز باقیمانده</th>
                                    <th class="text-right pb-2">حجم مصرفی</th>
                                    <th class="text-center pb-2">عملیات</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${user.configs.map(config => `
                                    <tr class="border-b border-white/5">
                                        <td class="py-2">${config.name}</td>
                                        <td class="py-2"><span class="px-2 py-0.5 rounded text-xs bg-primary-500/20 text-primary-400">${config.type}</span></td>
                                        <td class="py-2">${config.location}</td>
                                        <td class="py-2">${config.days_left} روز</td>
                                        <td class="py-2">${config.used_traffic} / ${config.total_traffic}</td>
                                        <td class="py-2 text-center">
                                            <button onclick="extendConfig(${config.id})" class="text-green-400 hover:text-green-300 mx-1" title="تمدید">
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                                                </svg>
                                            </button>
                                            <button onclick="revokeConfig(${config.id})" class="text-red-400 hover:text-red-300 mx-1" title="لغو">
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                                </svg>
                                            </button>
                                        </td>
                                    </tr>
                                `).join('') || '<tr><td colspan="6" class="text-center py-4 text-gray-400">کانفیگ فعالی ندارد</td></tr>'}
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <!-- دکمه‌های عملیات -->
                <div class="flex flex-wrap gap-3 mt-6">
                    <button onclick="editUser(${user.id}); closeUserDetailModal();" class="glass rounded-xl px-4 py-2 hover:bg-white/10 flex items-center gap-2">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                        </svg>
                        ویرایش
                    </button>
                    <button onclick="openBalanceModal(${user.id}, '${user.username}', ${user.balance}); closeUserDetailModal();" class="glass rounded-xl px-4 py-2 hover:bg-white/10 flex items-center gap-2">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                        تغییر موجودی
                    </button>
                    <button onclick="sendMessageToUser(${user.id})" class="glass rounded-xl px-4 py-2 hover:bg-white/10 flex items-center gap-2">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"></path>
                        </svg>
                        ارسال پیام
                    </button>
                    <button onclick="loginAsUser(${user.id})" class="glass rounded-xl px-4 py-2 hover:bg-white/10 flex items-center gap-2 text-yellow-400">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 16l-4-4m0 0l4-4m-4 4h14m-5 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h7a3 3 0 013 3v1"></path>
                        </svg>
                        ورود به حساب
                    </button>
                </div>
            `;
            
            document.getElementById('userDetailModal').classList.remove('hidden');
            document.getElementById('userDetailModal').classList.add('flex');
        }
    } catch (error) {
        showToast('خطا در دریافت جزئیات کاربر', 'error');
    }
}

function closeUserDetailModal() {
    document.getElementById('userDetailModal').classList.add('hidden');
    document.getElementById('userDetailModal').classList.remove('flex');
}

// Modal موجودی
function openBalanceModal(userId, username, balance) {
    document.getElementById('balanceUserId').value = userId;
    document.getElementById('balanceUsername').textContent = username;
    document.getElementById('currentBalance').textContent = numberFormat(balance);
    document.getElementById('balanceForm').reset();
    document.getElementById('balanceUserId').value = userId;
    document.getElementById('balanceModal').classList.remove('hidden');
    document.getElementById('balanceModal').classList.add('flex');
}

function closeBalanceModal() {
    document.getElementById('balanceModal').classList.add('hidden');
    document.getElementById('balanceModal').classList.remove('flex');
}

// ذخیره تغییر موجودی
document.getElementById('balanceForm')?.addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', 'admin_update_balance');
    
    try {
        const response = await fetch('api.php', { method: 'POST', body: formData });
        const data = await response.json();
        
        if (data.success) {
            showToast(data.message, 'success');
            closeBalanceModal();
            loadUsers(currentUserPage);
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در تغییر موجودی', 'error');
    }
});

// Modal اختصاص کانفیگ
async function openAssignConfigModal(userId, username) {
    document.getElementById('assignConfigUserId').value = userId;
    document.getElementById('assignConfigUsername').textContent = username;
    
    // لود لیست کانفیگ‌های موجود
    try {
        const response = await fetch('api.php?action=admin_get_available_configs');
        const data = await response.json();
        
        if (data.success) {
            const select = document.getElementById('configSelect');
            select.innerHTML = '<option value="">انتخاب کنید...</option>' + 
                data.configs.map(c => `<option value="${c.id}">${c.name} - ${c.type} (${c.location})</option>`).join('');
        }
    } catch (error) {
        console.error('Error loading configs:', error);
    }
    
    document.getElementById('assignConfigModal').classList.remove('hidden');
    document.getElementById('assignConfigModal').classList.add('flex');
}

function closeAssignConfigModal() {
    document.getElementById('assignConfigModal').classList.add('hidden');
    document.getElementById('assignConfigModal').classList.remove('flex');
}

// اختصاص کانفیگ
document.getElementById('assignConfigForm')?.addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', 'admin_assign_config');
    
    try {
        const response = await fetch('api.php', { method: 'POST', body: formData });
        const data = await response.json();
        
        if (data.success) {
            showToast(data.message, 'success');
            closeAssignConfigModal();
            loadUsers(currentUserPage);
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در اختصاص کانفیگ', 'error');
    }
});

// تغییر وضعیت کاربر
async function toggleUserStatus(userId, currentStatus) {
    const newStatus = currentStatus === 'blocked' ? 'active' : 'blocked';
    const action = newStatus === 'blocked' ? 'مسدود' : 'فعال';
    
    if (!confirm(`آیا از ${action} کردن این کاربر اطمینان دارید؟`)) return;
    
    try {
        const response = await fetch('api.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ action: 'admin_toggle_user_status', user_id: userId, status: newStatus })
        });
        const data = await response.json();
        
        if (data.success) {
            showToast(`کاربر ${action} شد`, 'success');
            loadUsers(currentUserPage);
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در تغییر وضعیت', 'error');
    }
}

// توابع کمکی
function getRoleGradient(role) {
    const gradients = {
        'user': 'from-blue-500 to-cyan-500',
        'vip': 'from-yellow-500 to-orange-500',
        'reseller': 'from-purple-500 to-pink-500',
        'admin': 'from-red-500 to-rose-500'
    };
    return gradients[role] || gradients['user'];
}

function getUserStatusClass(status) {
    const classes = {
        'active': 'bg-green-500/20 text-green-400',
        'blocked': 'bg-red-500/20 text-red-400',
        'pending': 'bg-yellow-500/20 text-yellow-400'
    };
    return classes[status] || classes['active'];
}

function getUserStatusText(status) {
    const texts = { 'active': 'فعال', 'blocked': 'مسدود', 'pending': 'در انتظار' };
    return texts[status] || status;
}

// Event Listeners
document.getElementById('userStatusFilter')?.addEventListener('change', () => loadUsers(1));
document.getElementById('userSortFilter')?.addEventListener('change', () => loadUsers(1));
document.getElementById('userRoleFilter')?.addEventListener('change', () => loadUsers(1));
document.getElementById('searchUsers')?.addEventListener('input', debounce(() => loadUsers(1), 500));

// لود اولیه
document.addEventListener('DOMContentLoaded', () => loadUsers());
</script>
<?php
}
?>