<?php
/**
 * مدیریت نوتیفیکیشن‌ها - پنل ادمین
 */
function renderAdminNotifications() {
?>
<main class="pr-0 lg:pr-64 pt-16 min-h-screen">
    <div class="p-4 lg:p-8">
        <!-- هدر صفحه -->
        <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4 mb-8">
            <div>
                <h1 class="text-2xl font-bold">مدیریت اعلان‌ها</h1>
                <p class="text-gray-400 mt-1">ارسال پیام و نوتیفیکیشن به کاربران</p>
            </div>
            <button onclick="openSendNotificationModal()" class="bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl px-6 py-2.5 font-medium hover:opacity-90 transition-opacity flex items-center gap-2">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                </svg>
                ارسال اعلان جدید
            </button>
        </div>

        <!-- آمار -->
        <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
            <div class="glass rounded-2xl p-4 text-center">
                <div class="text-3xl font-bold text-blue-400" id="totalNotifications">0</div>
                <div class="text-sm text-gray-400">کل اعلان‌ها</div>
            </div>
            <div class="glass rounded-2xl p-4 text-center">
                <div class="text-3xl font-bold text-green-400" id="sentToday">0</div>
                <div class="text-sm text-gray-400">ارسال امروز</div>
            </div>
            <div class="glass rounded-2xl p-4 text-center">
                <div class="text-3xl font-bold text-yellow-400" id="unreadCount">0</div>
                <div class="text-sm text-gray-400">خوانده نشده</div>
            </div>
            <div class="glass rounded-2xl p-4 text-center">
                <div class="text-3xl font-bold text-purple-400" id="broadcastCount">0</div>
                <div class="text-sm text-gray-400">پیام عمومی</div>
            </div>
        </div>

        <!-- تب‌ها -->
        <div class="flex items-center gap-2 mb-6">
            <button onclick="showNotificationTab('all')" class="notif-tab active px-6 py-2.5 rounded-xl bg-primary-500 text-white font-medium" data-tab="all">
                همه اعلان‌ها
            </button>
            <button onclick="showNotificationTab('broadcast')" class="notif-tab px-6 py-2.5 rounded-xl glass hover:bg-white/10 font-medium" data-tab="broadcast">
                پیام‌های عمومی
            </button>
            <button onclick="showNotificationTab('personal')" class="notif-tab px-6 py-2.5 rounded-xl glass hover:bg-white/10 font-medium" data-tab="personal">
                پیام‌های شخصی
            </button>
        </div>

        <!-- فیلترها -->
        <div class="glass rounded-2xl p-4 mb-6">
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div class="relative">
                    <input type="text" id="searchNotifications" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 pr-12 focus:border-primary-500 outline-none" placeholder="جستجو...">
                    <svg class="w-5 h-5 absolute right-4 top-1/2 -translate-y-1/2 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                    </svg>
                </div>
                <select id="notifTypeFilter" class="bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <option value="">همه انواع</option>
                    <option value="info">اطلاع‌رسانی</option>
                    <option value="success">موفقیت</option>
                    <option value="warning">هشدار</option>
                    <option value="danger">خطر</option>
                </select>
                <select id="notifStatusFilter" class="bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <option value="">همه وضعیت‌ها</option>
                    <option value="read">خوانده شده</option>
                    <option value="unread">خوانده نشده</option>
                </select>
                <input type="date" id="notifDateFilter" class="bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
            </div>
        </div>

        <!-- جدول اعلان‌ها -->
        <div class="glass rounded-3xl overflow-hidden">
            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead>
                        <tr class="text-gray-400 text-sm border-b border-white/10">
                            <th class="text-right p-4">
                                <input type="checkbox" id="selectAllNotifs" class="rounded bg-white/10 border-white/20">
                            </th>
                            <th class="text-right p-4">عنوان</th>
                            <th class="text-right p-4">گیرنده</th>
                            <th class="text-right p-4">نوع</th>
                            <th class="text-right p-4">وضعیت</th>
                            <th class="text-right p-4">تاریخ</th>
                            <th class="text-center p-4">عملیات</th>
                        </tr>
                    </thead>
                    <tbody id="notificationsTable">
                        <!-- داده‌ها با AJAX لود می‌شوند -->
                    </tbody>
                </table>
            </div>
            
            <!-- صفحه‌بندی -->
            <div class="p-4 border-t border-white/10">
                <div class="flex items-center justify-between">
                    <div class="flex items-center gap-2">
                        <span class="text-gray-400 text-sm" id="notifsCount">0 اعلان</span>
                        <button onclick="bulkDeleteNotifications()" class="px-3 py-1.5 rounded-lg bg-red-500/20 text-red-400 text-sm hover:bg-red-500/30">
                            حذف انتخاب شده
                        </button>
                    </div>
                    <div class="flex items-center gap-2" id="notifsPagination"></div>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- Modal ارسال اعلان -->
<div id="sendNotificationModal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 bg-black/50 backdrop-blur-sm">
    <div class="glass rounded-3xl p-6 w-full max-w-lg animate-scale-in max-h-[90vh] overflow-y-auto">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-lg font-bold">ارسال اعلان جدید</h3>
            <button onclick="closeSendNotificationModal()" class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-500/20">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        
        <form id="sendNotificationForm" class="space-y-4">
            <!-- نوع گیرنده -->
            <div>
                <label class="block text-sm text-gray-400 mb-2">گیرنده</label>
                <div class="grid grid-cols-3 gap-3">
                    <label class="glass rounded-xl p-3 cursor-pointer border-2 border-transparent has-[:checked]:border-primary-500 text-center">
                        <input type="radio" name="recipient_type" value="all" class="hidden" checked>
                        <div class="text-2xl mb-1">👥</div>
                        <div class="text-sm">همه کاربران</div>
                    </label>
                    <label class="glass rounded-xl p-3 cursor-pointer border-2 border-transparent has-[:checked]:border-primary-500 text-center">
                        <input type="radio" name="recipient_type" value="specific" class="hidden">
                        <div class="text-2xl mb-1">👤</div>
                        <div class="text-sm">کاربر خاص</div>
                    </label>
                    <label class="glass rounded-xl p-3 cursor-pointer border-2 border-transparent has-[:checked]:border-primary-500 text-center">
                        <input type="radio" name="recipient_type" value="group" class="hidden">
                        <div class="text-2xl mb-1">👨‍👩‍👧‍👦</div>
                        <div class="text-sm">گروه</div>
                    </label>
                </div>
            </div>
            
            <!-- انتخاب کاربر خاص -->
            <div id="specificUserSelect" class="hidden">
                <label class="block text-sm text-gray-400 mb-2">انتخاب کاربر</label>
                <select name="user_id" id="notifUserSelect" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <option value="">جستجو و انتخاب کاربر...</option>
                </select>
            </div>
            
            <!-- انتخاب گروه -->
            <div id="groupSelect" class="hidden">
                <label class="block text-sm text-gray-400 mb-2">انتخاب گروه</label>
                <select name="group" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <option value="active">کاربران فعال</option>
                    <option value="vip">کاربران VIP</option>
                    <option value="new">کاربران جدید (7 روز اخیر)</option>
                    <option value="inactive">کاربران غیرفعال</option>
                    <option value="expiring">کانفیگ در حال انقضا</option>
                </select>
            </div>
            
            <!-- نوع اعلان -->
            <div>
                <label class="block text-sm text-gray-400 mb-2">نوع اعلان</label>
                <div class="grid grid-cols-4 gap-2">
                    <label class="glass rounded-xl p-2 cursor-pointer border-2 border-transparent has-[:checked]:border-blue-500 text-center">
                        <input type="radio" name="type" value="info" class="hidden" checked>
                        <div class="text-blue-400">ℹ️</div>
                        <div class="text-xs mt-1">اطلاع</div>
                    </label>
                    <label class="glass rounded-xl p-2 cursor-pointer border-2 border-transparent has-[:checked]:border-green-500 text-center">
                        <input type="radio" name="type" value="success" class="hidden">
                        <div class="text-green-400">✅</div>
                        <div class="text-xs mt-1">موفقیت</div>
                    </label>
                    <label class="glass rounded-xl p-2 cursor-pointer border-2 border-transparent has-[:checked]:border-yellow-500 text-center">
                        <input type="radio" name="type" value="warning" class="hidden">
                        <div class="text-yellow-400">⚠️</div>
                        <div class="text-xs mt-1">هشدار</div>
                    </label>
                    <label class="glass rounded-xl p-2 cursor-pointer border-2 border-transparent has-[:checked]:border-red-500 text-center">
                        <input type="radio" name="type" value="danger" class="hidden">
                        <div class="text-red-400">🚨</div>
                        <div class="text-xs mt-1">خطر</div>
                    </label>
                </div>
            </div>
            
            <!-- عنوان -->
            <div>
                <label class="block text-sm text-gray-400 mb-2">عنوان</label>
                <input type="text" name="title" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" placeholder="عنوان اعلان" required>
            </div>
            
            <!-- متن پیام -->
            <div>
                <label class="block text-sm text-gray-400 mb-2">متن پیام</label>
                <textarea name="message" rows="4" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none resize-none" placeholder="متن اعلان..." required></textarea>
            </div>
            
            <!-- لینک (اختیاری) -->
            <div>
                <label class="block text-sm text-gray-400 mb-2">لینک (اختیاری)</label>
                <input type="url" name="link" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" placeholder="https://...">
            </div>
            
            <!-- ارسال همزمان به تلگرام -->
            <label class="flex items-center gap-3 p-3 bg-white/5 rounded-xl cursor-pointer">
                <input type="checkbox" name="send_telegram" class="rounded bg-white/10 border-white/20 text-primary-500">
                <div>
                    <div class="font-medium">ارسال به تلگرام</div>
                    <div class="text-xs text-gray-400">اعلان همزمان به ربات تلگرام ارسال شود</div>
                </div>
            </label>
            
            <div class="flex gap-3 pt-4">
                <button type="submit" class="flex-1 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl py-3 font-medium hover:opacity-90 transition-opacity">
                    ارسال اعلان
                </button>
                <button type="button" onclick="closeSendNotificationModal()" class="flex-1 glass rounded-xl py-3 font-medium hover:bg-white/10">
                    انصراف
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Modal مشاهده اعلان -->
<div id="viewNotificationModal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 bg-black/50 backdrop-blur-sm">
    <div class="glass rounded-3xl p-6 w-full max-w-md animate-scale-in">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-lg font-bold" id="viewNotifTitle">عنوان اعلان</h3>
            <button onclick="closeViewNotificationModal()" class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-500/20">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        <div id="viewNotifContent" class="space-y-4">
            <!-- محتوا با AJAX لود می‌شود -->
        </div>
    </div>
</div>

<script>
let currentNotifTab = 'all';
let currentNotifPage = 1;

// نمایش تب
function showNotificationTab(tab) {
    currentNotifTab = tab;
    document.querySelectorAll('.notif-tab').forEach(t => {
        if (t.dataset.tab === tab) {
            t.classList.add('active', 'bg-primary-500', 'text-white');
            t.classList.remove('glass');
        } else {
            t.classList.remove('active', 'bg-primary-500', 'text-white');
            t.classList.add('glass');
        }
    });
    loadNotifications(1);
}

// لود اعلان‌ها
async function loadNotifications(page = 1) {
    currentNotifPage = page;
    const type = document.getElementById('notifTypeFilter').value;
    const status = document.getElementById('notifStatusFilter').value;
    const search = document.getElementById('searchNotifications').value;
    const date = document.getElementById('notifDateFilter').value;
    
    try {
        const response = await fetch(`api.php?action=admin_get_notifications&tab=${currentNotifTab}&type=${type}&status=${status}&search=${encodeURIComponent(search)}&date=${date}&page=${page}`);
        const data = await response.json();
        
        if (data.success) {
            renderNotificationsTable(data.notifications);
            document.getElementById('notifsCount').textContent = `${data.total} اعلان`;
            
            // آمار
            document.getElementById('totalNotifications').textContent = data.stats.total;
            document.getElementById('sentToday').textContent = data.stats.today;
            document.getElementById('unreadCount').textContent = data.stats.unread;
            document.getElementById('broadcastCount').textContent = data.stats.broadcast;
        }
    } catch (error) {
        console.error('Error loading notifications:', error);
    }
}

function renderNotificationsTable(notifications) {
    const tbody = document.getElementById('notificationsTable');
    
    if (notifications.length === 0) {
        tbody.innerHTML = `
            <tr>
                <td colspan="7" class="text-center py-12 text-gray-400">
                    <svg class="w-16 h-16 mx-auto mb-4 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"></path>
                    </svg>
                    اعلانی یافت نشد
                </td>
            </tr>
        `;
        return;
    }
    
    tbody.innerHTML = notifications.map(notif => `
        <tr class="border-b border-white/5 hover:bg-white/5">
            <td class="p-4">
                <input type="checkbox" class="notif-checkbox rounded bg-white/10 border-white/20" data-id="${notif.id}">
            </td>
            <td class="p-4">
                <div class="flex items-center gap-3">
                    <div class="w-10 h-10 rounded-xl ${getNotifTypeClass(notif.type)} flex items-center justify-center">
                        ${getNotifTypeIcon(notif.type)}
                    </div>
                    <div>
                        <div class="font-medium">${notif.title}</div>
                        <div class="text-sm text-gray-400 truncate max-w-xs">${notif.message.substring(0, 50)}...</div>
                    </div>
                </div>
            </td>
            <td class="p-4">
                ${notif.user_id ? `
                    <span class="text-primary-400">${notif.username}</span>
                ` : `
                    <span class="px-2 py-1 rounded-lg bg-purple-500/20 text-purple-400 text-xs">همه کاربران</span>
                `}
            </td>
            <td class="p-4">
                <span class="px-2 py-1 rounded-lg text-xs ${getNotifTypeBadge(notif.type)}">
                    ${getNotifTypeText(notif.type)}
                </span>
            </td>
            <td class="p-4">
                <span class="px-2 py-1 rounded-lg text-xs ${notif.is_read ? 'bg-gray-500/20 text-gray-400' : 'bg-blue-500/20 text-blue-400'}">
                    ${notif.is_read ? 'خوانده شده' : 'خوانده نشده'}
                </span>
            </td>
            <td class="p-4 text-gray-400 text-sm">${notif.created_at}</td>
            <td class="p-4">
                <div class="flex items-center justify-center gap-2">
                    <button onclick="viewNotification(${notif.id})" class="w-8 h-8 rounded-lg bg-blue-500/20 text-blue-400 flex items-center justify-center hover:bg-blue-500/30" title="مشاهده">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path>
                        </svg>
                    </button>
                    <button onclick="resendNotification(${notif.id})" class="w-8 h-8 rounded-lg bg-green-500/20 text-green-400 flex items-center justify-center hover:bg-green-500/30" title="ارسال مجدد">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                        </svg>
                    </button>
                    <button onclick="deleteNotification(${notif.id})" class="w-8 h-8 rounded-lg bg-red-500/20 text-red-400 flex items-center justify-center hover:bg-red-500/30" title="حذف">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                        </svg>
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Modal ارسال اعلان
function openSendNotificationModal() {
    document.getElementById('sendNotificationForm').reset();
    loadUsersForNotification();
    document.getElementById('sendNotificationModal').classList.remove('hidden');
    document.getElementById('sendNotificationModal').classList.add('flex');
}

function closeSendNotificationModal() {
    document.getElementById('sendNotificationModal').classList.add('hidden');
    document.getElementById('sendNotificationModal').classList.remove('flex');
}

// لود کاربران
async function loadUsersForNotification() {
    try {
        const response = await fetch('api.php?action=admin_get_users_list');
        const data = await response.json();
        
        if (data.success) {
            const select = document.getElementById('notifUserSelect');
            select.innerHTML = '<option value="">انتخاب کنید...</option>' + 
                data.users.map(u => `<option value="${u.id}">${u.username} - ${u.phone}</option>`).join('');
        }
    } catch (error) {
        console.error('Error loading users:', error);
    }
}

// نمایش/مخفی فیلدها بر اساس نوع گیرنده
document.querySelectorAll('input[name="recipient_type"]').forEach(radio => {
    radio.addEventListener('change', function() {
        document.getElementById('specificUserSelect').classList.toggle('hidden', this.value !== 'specific');
        document.getElementById('groupSelect').classList.toggle('hidden', this.value !== 'group');
    });
});

// ارسال اعلان
document.getElementById('sendNotificationForm')?.addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', 'admin_send_notification');
    
    try {
        const response = await fetch('api.php', { method: 'POST', body: formData });
        const data = await response.json();
        
        if (data.success) {
            showToast(`اعلان به ${data.count} کاربر ارسال شد`, 'success');
            closeSendNotificationModal();
            loadNotifications();
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در ارسال اعلان', 'error');
    }
});

// مشاهده اعلان
async function viewNotification(id) {
    try {
        const response = await fetch(`api.php?action=admin_get_notification&id=${id}`);
        const data = await response.json();
        
        if (data.success) {
            const notif = data.notification;
            document.getElementById('viewNotifTitle').textContent = notif.title;
            document.getElementById('viewNotifContent').innerHTML = `
                <div class="p-4 rounded-xl ${getNotifTypeClass(notif.type)}">
                    <div class="flex items-center gap-2 mb-2">
                        ${getNotifTypeIcon(notif.type)}
                        <span class="font-medium">${getNotifTypeText(notif.type)}</span>
                    </div>
                    <p class="text-sm">${notif.message}</p>
                </div>
                <div class="grid grid-cols-2 gap-4 text-sm">
                    <div class="glass rounded-xl p-3">
                        <div class="text-gray-400">گیرنده</div>
                        <div class="font-medium">${notif.username || 'همه کاربران'}</div>
                    </div>
                    <div class="glass rounded-xl p-3">
                        <div class="text-gray-400">تاریخ ارسال</div>
                        <div class="font-medium">${notif.created_at}</div>
                    </div>
                </div>
                ${notif.link ? `
                    <a href="${notif.link}" target="_blank" class="block glass rounded-xl p-3 text-primary-400 hover:bg-white/10">
                        🔗 ${notif.link}
                    </a>
                ` : ''}
            `;
            
            document.getElementById('viewNotificationModal').classList.remove('hidden');
            document.getElementById('viewNotificationModal').classList.add('flex');
        }
    } catch (error) {
        showToast('خطا در دریافت اعلان', 'error');
    }
}

function closeViewNotificationModal() {
    document.getElementById('viewNotificationModal').classList.add('hidden');
    document.getElementById('viewNotificationModal').classList.remove('flex');
}

// حذف اعلان
async function deleteNotification(id) {
    if (!confirm('آیا از حذف این اعلان اطمینان دارید؟')) return;
    
    try {
        const response = await fetch('api.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ action: 'admin_delete_notification', id: id })
        });
        const data = await response.json();
        
        if (data.success) {
            showToast('اعلان حذف شد', 'success');
            loadNotifications(currentNotifPage);
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در حذف اعلان', 'error');
    }
}

// توابع کمکی
function getNotifTypeClass(type) {
    const classes = {
        'info': 'bg-blue-500/20',
        'success': 'bg-green-500/20',
        'warning': 'bg-yellow-500/20',
        'danger': 'bg-red-500/20'
    };
    return classes[type] || classes['info'];
}

function getNotifTypeIcon(type) {
    const icons = {
        'info': '<span class="text-blue-400">ℹ️</span>',
        'success': '<span class="text-green-400">✅</span>',
        'warning': '<span class="text-yellow-400">⚠️</span>',
        'danger': '<span class="text-red-400">🚨</span>'
    };
    return icons[type] || icons['info'];
}

function getNotifTypeBadge(type) {
    const classes = {
        'info': 'bg-blue-500/20 text-blue-400',
        'success': 'bg-green-500/20 text-green-400',
        'warning': 'bg-yellow-500/20 text-yellow-400',
        'danger': 'bg-red-500/20 text-red-400'
    };
    return classes[type] || classes['info'];
}

function getNotifTypeText(type) {
    const texts = { 'info': 'اطلاع‌رسانی', 'success': 'موفقیت', 'warning': 'هشدار', 'danger': 'خطر' };
    return texts[type] || type;
}

// Event Listeners
document.getElementById('notifTypeFilter')?.addEventListener('change', () => loadNotifications(1));
document.getElementById('notifStatusFilter')?.addEventListener('change', () => loadNotifications(1));
document.getElementById('notifDateFilter')?.addEventListener('change', () => loadNotifications(1));
document.getElementById('searchNotifications')?.addEventListener('input', debounce(() => loadNotifications(1), 500));

// لود اولیه
document.addEventListener('DOMContentLoaded', () => loadNotifications());
</script>
<?php
}
?>