<?php
/**
 * توابع اصلی سیستم
 */

require_once 'config.php';

// ============================================
// توابع کاربران
// ============================================

function getUserById($id) {
    return db()->fetch("SELECT * FROM users WHERE id = ?", [$id]);
}

function getUserByUsername($username) {
    return db()->fetch("SELECT * FROM users WHERE username = ? OR phone = ?", [$username, $username]);
}

function getUserByPhone($phone) {
    return db()->fetch("SELECT * FROM users WHERE phone = ?", [$phone]);
}

function getUserByReferralCode($code) {
    return db()->fetch("SELECT * FROM users WHERE referral_code = ?", [$code]);
}

function createUser($data) {
    return db()->insert('users', $data);
}

function updateUser($id, $data) {
    return db()->update('users', $data, 'id = ?', [$id]);
}

function updateLastLogin($id) {
    return db()->update('users', [
        'last_login' => date('Y-m-d H:i:s'),
        'last_ip' => $_SERVER['REMOTE_ADDR'] ?? ''
    ], 'id = ?', [$id]);
}

function updateUserBalance($userId, $amount) {
    $sql = "UPDATE users SET balance = balance + ? WHERE id = ?";
    return db()->query($sql, [$amount, $userId]);
}

function getAllUsers($page = 1, $limit = ITEMS_PER_PAGE, $filters = []) {
    $offset = ($page - 1) * $limit;
    $where = "1=1";
    $params = [];
    
    if (!empty($filters['status'])) {
        $where .= " AND status = ?";
        $params[] = $filters['status'];
    }
    
    if (!empty($filters['role'])) {
        $where .= " AND role = ?";
        $params[] = $filters['role'];
    }
    
    if (!empty($filters['search'])) {
        $where .= " AND (username LIKE ? OR phone LIKE ? OR email LIKE ?)";
        $search = '%' . $filters['search'] . '%';
        $params = array_merge($params, [$search, $search, $search]);
    }
    
    $sort = match($filters['sort'] ?? 'newest') {
        'oldest' => 'created_at ASC',
        'most_orders' => '(SELECT COUNT(*) FROM orders WHERE orders.user_id = users.id) DESC',
        'highest_balance' => 'balance DESC',
        default => 'created_at DESC'
    };
    
    $sql = "SELECT u.*, 
            (SELECT COUNT(*) FROM configs WHERE assigned_to = u.id AND status = 'sold') as active_configs
            FROM users u WHERE {$where} ORDER BY {$sort} LIMIT {$limit} OFFSET {$offset}";
    
    return db()->fetchAll($sql, $params);
}

function getUsersCount($filters = []) {
    $where = "1=1";
    $params = [];
    
    if (!empty($filters['status'])) {
        $where .= " AND status = ?";
        $params[] = $filters['status'];
    }
    
    if (!empty($filters['search'])) {
        $where .= " AND (username LIKE ? OR phone LIKE ?)";
        $search = '%' . $filters['search'] . '%';
        $params = array_merge($params, [$search, $search]);
    }
    
    return db()->fetch("SELECT COUNT(*) as count FROM users WHERE {$where}", $params)['count'];
}

function getUserStats() {
    return [
        'total' => db()->fetch("SELECT COUNT(*) as c FROM users")['c'],
        'active' => db()->fetch("SELECT COUNT(*) as c FROM users WHERE status = 'active'")['c'],
        'blocked' => db()->fetch("SELECT COUNT(*) as c FROM users WHERE status = 'blocked'")['c'],
        'today' => db()->fetch("SELECT COUNT(*) as c FROM users WHERE DATE(created_at) = CURDATE()")['c']
    ];
}

// ============================================
// توابع محصولات
// ============================================

function getProducts($type = 'all', $location = '', $sort = 'popular') {
    $where = "is_active = 1 AND stock > 0";
    $params = [];
    
    if ($type !== 'all') {
        $where .= " AND type = ?";
        $params[] = $type;
    }
    
    if ($location) {
        $where .= " AND location = ?";
        $params[] = $location;
    }
    
    $orderBy = match($sort) {
        'price_asc' => 'price ASC',
        'price_desc' => 'price DESC',
        'newest' => 'created_at DESC',
        default => 'sold_count DESC, is_featured DESC'
    };
    
    $sql = "SELECT * FROM products WHERE {$where} ORDER BY {$orderBy}";
    return db()->fetchAll($sql, $params);
}

function getProductById($id) {
    return db()->fetch("SELECT * FROM products WHERE id = ?", [$id]);
}

function createProduct($data) {
    return db()->insert('products', $data);
}

function updateProduct($id, $data) {
    return db()->update('products', $data, 'id = ?', [$id]);
}

function deleteProduct($id) {
    return db()->delete('products', 'id = ?', [$id]);
}

// ============================================
// توابع کانفیگ
// ============================================

function getAvailableConfig($productId) {
    return db()->fetch(
        "SELECT * FROM configs WHERE product_id = ? AND status = 'available' LIMIT 1",
        [$productId]
    );
}

function assignConfigToUser($userId, $productId, $orderId) {
    $config = getAvailableConfig($productId);
    
    if (!$config) {
        // ساخت کانفیگ جدید اگر موجود نبود
        return false;
    }
    
    $product = getProductById($productId);
    $expiresAt = date('Y-m-d H:i:s', strtotime("+{$product['duration']} days"));
    
    db()->update('configs', [
        'status' => 'sold',
        'assigned_to' => $userId,
        'assigned_at' => date('Y-m-d H:i:s'),
        'expires_at' => $expiresAt
    ], 'id = ?', [$config['id']]);
    
    // بروزرسانی سفارش
    db()->update('orders', ['config_id' => $config['id']], 'id = ?', [$orderId]);
    
    // بروزرسانی تعداد فروش محصول
    db()->query("UPDATE products SET sold_count = sold_count + 1, stock = stock - 1 WHERE id = ?", [$productId]);
    
    return $config['id'];
}

function getUserActiveConfigs($userId) {
    $sql = "SELECT c.*, p.name, p.type, p.location, p.traffic,
            DATEDIFF(c.expires_at, NOW()) as days_left,
            ROUND(c.used_traffic / 1073741824, 2) as used_traffic_gb
            FROM configs c
            JOIN products p ON c.product_id = p.id
            WHERE c.assigned_to = ? AND c.status = 'sold' AND c.expires_at > NOW()
            ORDER BY c.expires_at ASC";
    
    $configs = db()->fetchAll($sql, [$userId]);
    
    return array_map(function($config) {
        $config['days_left'] = max(0, $config['days_left']);
        $config['used_traffic'] = $config['used_traffic_gb'] ?? 0;
        $config['traffic_limit'] = $config['traffic'];
        return $config;
    }, $configs);
}

function getUserConfig($userId, $configId) {
    return db()->fetch(
        "SELECT * FROM configs WHERE id = ? AND assigned_to = ?",
        [$configId, $userId]
    );
}

// ============================================
// توابع سفارشات
// ============================================

function createOrder($data) {
    return db()->insert('orders', $data);
}

function getOrderById($id) {
    return db()->fetch("SELECT * FROM orders WHERE id = ?", [$id]);
}

function getOrders($page = 1, $limit = ITEMS_PER_PAGE, $filters = []) {
    $offset = ($page - 1) * $limit;
    $where = "1=1";
    $params = [];
    
    if (!empty($filters['status'])) {
        $where .= " AND o.status = ?";
        $params[] = $filters['status'];
    }
    
    if (!empty($filters['payment_method'])) {
        $where .= " AND o.payment_method = ?";
        $params[] = $filters['payment_method'];
    }
    
    if (!empty($filters['user_id'])) {
        $where .= " AND o.user_id = ?";
        $params[] = $filters['user_id'];
    }
    
    $sql = "SELECT o.*, u.username, u.phone, p.name as product_name, p.type as product_type
            FROM orders o
            JOIN users u ON o.user_id = u.id
            JOIN products p ON o.product_id = p.id
            WHERE {$where}
            ORDER BY o.created_at DESC
            LIMIT {$limit} OFFSET {$offset}";
    
    $orders = db()->fetchAll($sql, $params);
    
    return array_map(function($order) {
        $order['created_at'] = toJalali($order['created_at']);
        return $order;
    }, $orders);
}

function approveOrder($orderId) {
    $order = getOrderById($orderId);
    if (!$order || $order['status'] !== 'pending') {
        return false;
    }
    
    // تخصیص کانفیگ
    $configId = assignConfigToUser($order['user_id'], $order['product_id'], $orderId);
    
    // بروزرسانی وضعیت سفارش
    db()->update('orders', [
        'status' => 'completed',
        'completed_at' => date('Y-m-d H:i:s')
    ], 'id = ?', [$orderId]);
    
    // کمیسیون رفرال
    $user = getUserById($order['user_id']);
    if ($user['referred_by']) {
        $commission = floor($order['amount'] * getSetting('referral_commission', 10) / 100);
        if ($commission > 0) {
            updateUserBalance($user['referred_by'], $commission);
            createTransaction($user['referred_by'], $commission, 'referral', 'کمیسیون خرید زیرمجموعه');
            createReferralCommission($user['referred_by'], $user['id'], $orderId, $order['amount'], $commission);
        }
    }
    
    // ارسال نوتیفیکیشن
    createNotification($order['user_id'], 'سفارش تایید شد', 'سفارش شما تایید و کانفیگ فعال شد', 'success');
    
    return true;
}

function rejectOrder($orderId, $reason = '') {
    $order = getOrderById($orderId);
    if (!$order || $order['status'] !== 'pending') {
        return false;
    }
    
    db()->update('orders', [
        'status' => 'rejected',
        'reject_reason' => $reason
    ], 'id = ?', [$orderId]);
    
    // بازگشت وجه اگر از کیف پول پرداخت شده بود
    if ($order['payment_method'] === 'wallet') {
        updateUserBalance($order['user_id'], $order['amount']);
        createTransaction($order['user_id'], $order['amount'], 'refund', 'بازگشت وجه سفارش رد شده');
    }
    
    createNotification($order['user_id'], 'سفارش رد شد', $reason ?: 'سفارش شما رد شد', 'danger');
    
    return true;
}

// ============================================
// توابع تراکنش‌ها
// ============================================

function createTransaction($userId, $amount, $type, $description = '', $referenceId = null) {
    return db()->insert('transactions', [
        'user_id' => $userId,
        'amount' => $amount,
        'type' => $type,
        'description' => $description,
        'reference_id' => $referenceId
    ]);
}

function getUserTransactions($userId, $type = '', $page = 1, $limit = 20) {
    $offset = ($page - 1) * $limit;
    $where = "user_id = ?";
    $params = [$userId];
    
    if ($type) {
        $where .= " AND type = ?";
        $params[] = $type;
    }
    
    $sql = "SELECT * FROM transactions WHERE {$where} ORDER BY created_at DESC LIMIT {$limit} OFFSET {$offset}";
    $transactions = db()->fetchAll($sql, $params);
    
    return array_map(function($tx) {
        $tx['created_at'] = toJalali($tx['created_at']);
        return $tx;
    }, $transactions);
}

// ============================================
// توابع شارژ
// ============================================

function createChargeRequest($data) {
    return db()->insert('charge_requests', $data);
}

function approveChargeRequest($id) {
    $request = db()->fetch("SELECT * FROM charge_requests WHERE id = ? AND status = 'pending'", [$id]);
    if (!$request) return false;
    
    db()->update('charge_requests', [
        'status' => 'approved',
        'admin_id' => $_SESSION['user_id'],
        'processed_at' => date('Y-m-d H:i:s')
    ], 'id = ?', [$id]);
    
    updateUserBalance($request['user_id'], $request['amount']);
    createTransaction($request['user_id'], $request['amount'], 'charge', 'شارژ کیف پول');
    createNotification($request['user_id'], 'شارژ موفق', 'کیف پول شما شارژ شد', 'success');
    
    return true;
}

// ============================================
// توابع تخفیف
// ============================================

function getDiscountByCode($code) {
    return db()->fetch(
        "SELECT * FROM discounts WHERE code = ? AND is_active = 1",
        [$code]
    );
}

function useDiscount($code) {
    db()->query("UPDATE discounts SET used_count = used_count + 1 WHERE code = ?", [$code]);
}

// ============================================
// توابع تیکت
// ============================================

function createTicket($data) {
    return db()->insert('tickets', $data);
}

function createTicketMessage($data) {
    $messageId = db()->insert('ticket_messages', $data);
    db()->update('tickets', ['updated_at' => date('Y-m-d H:i:s')], 'id = ?', [$data['ticket_id']]);
    return $messageId;
}

function getTicketById($id) {
    return db()->fetch("SELECT * FROM tickets WHERE id = ?", [$id]);
}

function getUserTickets($userId) {
    $sql = "SELECT * FROM tickets WHERE user_id = ? ORDER BY updated_at DESC";
    $tickets = db()->fetchAll($sql, [$userId]);
    
    return array_map(function($ticket) {
        $ticket['created_at'] = toJalali($ticket['created_at']);
        $ticket['updated_at'] = toJalali($ticket['updated_at']);
        return $ticket;
    }, $tickets);
}

function getTicketWithMessages($ticketId, $userId = null) {
    $where = $userId ? "id = ? AND user_id = ?" : "id = ?";
    $params = $userId ? [$ticketId, $userId] : [$ticketId];
    
    $ticket = db()->fetch("SELECT * FROM tickets WHERE {$where}", $params);
    if (!$ticket) return null;
    
    $messages = db()->fetchAll(
        "SELECT tm.*, u.username FROM ticket_messages tm 
         JOIN users u ON tm.user_id = u.id 
         WHERE tm.ticket_id = ? ORDER BY tm.created_at ASC",
        [$ticketId]
    );
    
    $ticket['messages'] = array_map(function($msg) {
        $msg['created_at'] = toJalali($msg['created_at']);
        return $msg;
    }, $messages);
    
    $ticket['created_at'] = toJalali($ticket['created_at']);
    
    return $ticket;
}

function updateTicketStatus($ticketId, $status) {
    return db()->update('tickets', ['status' => $status], 'id = ?', [$ticketId]);
}

// ============================================
// توابع نوتیفیکیشن
// ============================================

function createNotification($userId, $title, $message, $type = 'info') {
    return db()->insert('notifications', [
        'user_id' => $userId,
        'title' => $title,
        'message' => $message,
        'type' => $type
    ]);
}

function getUserNotifications($userId, $unreadOnly = false) {
    $where = "user_id = ? OR user_id IS NULL";
    if ($unreadOnly) {
        $where .= " AND is_read = 0";
    }
    return db()->fetchAll(
        "SELECT * FROM notifications WHERE {$where} ORDER BY created_at DESC LIMIT 20",
        [$userId]
    );
}

// ============================================
// توابع رفرال
// ============================================

function getUserReferrals($userId) {
    return db()->fetchAll(
        "SELECT id, username, created_at FROM users WHERE referred_by = ?",
        [$userId]
    );
}

function getUserReferralReward($userId) {
    $result = db()->fetch(
        "SELECT SUM(commission_amount) as total FROM referral_commissions WHERE referrer_id = ?",
        [$userId]
    );
    return $result['total'] ?? 0;
}

function createReferralCommission($referrerId, $referredId, $orderId, $orderAmount, $commission) {
    return db()->insert('referral_commissions', [
        'referrer_id' => $referrerId,
        'referred_id' => $referredId,
        'order_id' => $orderId,
        'order_amount' => $orderAmount,
        'commission_amount' => $commission
    ]);
}

// ============================================
// توابع دستگاه
// ============================================

function getUserDevices($userId) {
    return db()->fetchAll(
        "SELECT * FROM devices WHERE user_id = ? ORDER BY last_connect DESC",
        [$userId]
    );
}

function registerDevice($userId, $configId, $deviceInfo) {
    $existing = db()->fetch(
        "SELECT id FROM devices WHERE user_id = ? AND config_id = ? AND device_name = ?",
        [$userId, $configId, $deviceInfo['name']]
    );
    
    if ($existing) {
        return db()->update('devices', [
            'last_connect' => date('Y-m-d H:i:s'),
            'ip_address' => $deviceInfo['ip']
        ], 'id = ?', [$existing['id']]);
    }
    
    // بررسی محدودیت دستگاه
    $user = getUserById($userId);
    $deviceCount = db()->fetch(
        "SELECT COUNT(*) as c FROM devices WHERE user_id = ? AND config_id = ?",
        [$userId, $configId]
    )['c'];
    
    if ($deviceCount >= $user['max_devices']) {
        return false;
    }
    
    return db()->insert('devices', [
        'user_id' => $userId,
        'config_id' => $configId,
        'device_name' => $deviceInfo['name'],
        'device_type' => $deviceInfo['type'],
        'ip_address' => $deviceInfo['ip'],
        'user_agent' => $deviceInfo['user_agent'],
        'last_connect' => date('Y-m-d H:i:s')
    ]);
}

// ============================================
// توابع تنظیمات
// ============================================

function getSetting($key, $default = null) {
    $result = db()->fetch("SELECT value FROM settings WHERE `key` = ?", [$key]);
    return $result ? $result['value'] : $default;
}

function setSetting($key, $value) {
    $existing = db()->fetch("SELECT id FROM settings WHERE `key` = ?", [$key]);
    
    if ($existing) {
        return db()->update('settings', ['value' => $value], '`key` = ?', [$key]);
    }
    
    return db()->insert('settings', ['key' => $key, 'value' => $value]);
}

function getAllSettings($group = null) {
    $where = $group ? "WHERE `group` = ?" : "";
    $params = $group ? [$group] : [];
    
    $rows = db()->fetchAll("SELECT `key`, value FROM settings {$where}", $params);
    
    $settings = [];
    foreach ($rows as $row) {
        $settings[$row['key']] = $row['value'];
    }
    return $settings;
}

// ============================================
// توابع آمار داشبورد
// ============================================

function getAdminDashboardStats() {
    return [
        'total_users' => db()->fetch("SELECT COUNT(*) as c FROM users WHERE role != 'admin'")['c'],
        'active_configs' => db()->fetch("SELECT COUNT(*) as c FROM configs WHERE status = 'sold' AND expires_at > NOW()")['c'],
        'today_income' => db()->fetch("SELECT COALESCE(SUM(amount), 0) as s FROM orders WHERE status = 'completed' AND DATE(completed_at) = CURDATE()")['s'],
        'pending_orders' => db()->fetch("SELECT COUNT(*) as c FROM orders WHERE status = 'pending'")['c'],
        'pending_tickets' => db()->fetch("SELECT COUNT(*) as c FROM tickets WHERE status IN ('open', 'waiting')")['c'],
        'pending_charges' => db()->fetch("SELECT COUNT(*) as c FROM charge_requests WHERE status = 'pending'")['c']
    ];
}